import 'package:cached_network_image/cached_network_image.dart';
import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:flutter_svg/svg.dart';
import 'package:gap/gap.dart';
import 'package:hive_flutter/hive_flutter.dart';
import 'package:ready_grocery/components/ecommerce/add_to_cart_bottom_sheet.dart';
import 'package:ready_grocery/components/ecommerce/increment_button.dart';
import 'package:ready_grocery/config/app_color.dart';
import 'package:ready_grocery/config/app_constants.dart';
import 'package:ready_grocery/config/app_text_style.dart';
import 'package:ready_grocery/config/theme.dart';
import 'package:ready_grocery/gen/assets.gen.dart';
import 'package:ready_grocery/models/eCommerce/cart/hive_cart_model.dart';
import 'package:ready_grocery/models/eCommerce/product/product.dart';
import 'package:ready_grocery/utils/global_function.dart';
import 'package:ready_grocery/utils/is_dark_mode.dart';

class ListProductCard extends StatelessWidget {
  final Product product;
  final void Function()? onTap;
  final void Function()? onTapRemove;
  final EdgeInsets? margin;
  const ListProductCard(
      {super.key,
      required this.product,
      required this.onTap,
      this.onTapRemove,
      this.margin});

  @override
  Widget build(BuildContext context) {
    bool isDark = isDarkMode();
    return ValueListenableBuilder<Box<HiveCartModel>>(
        valueListenable:
            Hive.box<HiveCartModel>(AppConstants.cartModelBox).listenable(),
        builder: (context, cartBox, _) {
          bool inCart = false;
          late int productQuantity = 0;
          late int index = 0;
          final cartItems = cartBox.values.toList();
          for (int i = 0; i < cartItems.length; i++) {
            final cartProduct = cartItems[i];
            if (cartProduct.productId == product.id) {
              inCart = true;
              productQuantity = cartProduct.productsQTY;
              index = i;
              break;
            }
          }
          return Container(
            margin:
                margin ?? EdgeInsets.symmetric(horizontal: 16.w, vertical: 5.h),
            decoration: BoxDecoration(
                color: GlobalFunction.getContainerColor(),
                borderRadius: BorderRadius.circular(8.0.r),
                border: isDark
                    ? Border.all(color: Colors.white.withValues(alpha: 0.2))
                    : null,
                boxShadow: [
                  BoxShadow(
                    color: Colors.black.withValues(alpha: 0.1),
                    spreadRadius: 0,
                    blurRadius: 4,
                    offset: const Offset(0, 1),
                  ),
                ]),
            // shape: RoundedRectangleBorder(
            //   borderRadius: BorderRadius.circular(8.r),
            // ),
            child: InkWell(
              borderRadius: BorderRadius.circular(8.r),
              onTap: onTap,
              child: SizedBox(
                height: MediaQuery.of(context).size.height / 6.15,
                child: Row(
                  children: [
                    _buildImageSection(context: context),
                    _buildDetailsSection(
                      context: context,
                      inCart: inCart,
                      productQuantity: productQuantity,
                      cartBox: cartBox,
                      index: index,
                      cartItems: cartItems,
                      onTapRemove: onTapRemove,
                    ),
                  ],
                ),
              ),
            ),
          );
        });
  }

  Widget _buildImageSection({required BuildContext context}) {
    return Flexible(
      flex: 2,
      fit: FlexFit.tight,
      child: Stack(
        children: [
          ClipRRect(
            borderRadius: BorderRadius.circular(8.r),
            child: CachedNetworkImage(
              placeholder: (context, url) =>
                  const Center(child: CircularProgressIndicator()),
              imageUrl: product.thumbnail,
              height: MediaQuery.of(context).size.height,
              width: MediaQuery.of(context).size.width,
              fit: BoxFit.cover,
            ),
          ),
          if (product.quantity == 0)
            Positioned.fill(
              child: Container(
                decoration: BoxDecoration(
                  borderRadius: BorderRadius.circular(5.r),
                  color: Colors.black.withValues(alpha: 0.6),
                ),
                child: Center(
                  child: Text(
                    'Out of Stock',
                    style: AppTextStyle(context).subTitle.copyWith(
                          color: colors(context).light,
                        ),
                  ),
                ),
              ),
            ),
          if (product.discountPercentage != 0)
            Positioned(
              top: 4.h,
              left: 4.w,
              child: Container(
                padding: EdgeInsets.symmetric(horizontal: 2.w, vertical: 1.h),
                decoration: BoxDecoration(
                  borderRadius: BorderRadius.circular(5.r),
                  color: EcommerceAppColor.red,
                ),
                child: Text(
                  '-${product.discountPercentage}%',
                  style: AppTextStyle(context).bodyTextSmall.copyWith(
                        fontSize: 12.sp,
                        fontWeight: FontWeight.w700,
                        color: colors(context).light,
                      ),
                ),
              ),
            )
        ],
      ),
    );
  }

  Widget _buildDetailsSection(
      {required BuildContext context,
      required bool inCart,
      required int productQuantity,
      required Box<HiveCartModel> cartBox,
      required int index,
      required List<HiveCartModel> cartItems,
      Function()? onTapRemove}) {
    return Flexible(
      flex: 3,
      fit: FlexFit.tight,
      child: Padding(
        padding: EdgeInsets.symmetric(horizontal: 11.w, vertical: 8.h),
        child: Stack(
          children: [
            Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              mainAxisAlignment: MainAxisAlignment.center,
              children: [
                Row(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Expanded(
                      child: Text(
                        '${product.name}\n',
                        style: AppTextStyle(context)
                            .bodyText
                            .copyWith(fontWeight: FontWeight.w500),
                        maxLines: 2,
                        overflow: TextOverflow.ellipsis,
                      ),
                    ),
                    if (onTapRemove != null)
                      GestureDetector(
                        behavior: HitTestBehavior.translucent,
                        onTap: onTapRemove,
                        child: SvgPicture.asset(Assets.svg.trash),
                      )
                  ],
                ),
                Gap(8.h),
                Row(
                  mainAxisAlignment: MainAxisAlignment.spaceBetween,
                  children: [
                    SizedBox(
                      child: Row(
                        children: [
                          Icon(
                            Icons.star_rounded,
                            size: 16.sp,
                            color: EcommerceAppColor.carrotOrange,
                          ),
                          Text(
                            product.rating.toString(),
                            style: AppTextStyle(context).bodyTextSmall.copyWith(
                                  fontWeight: FontWeight.w700,
                                ),
                          ),
                          Gap(5.w),
                          Text(
                            '(${product.totalReviews})',
                            style: AppTextStyle(context)
                                .bodyTextSmall
                                .copyWith(fontWeight: FontWeight.w500),
                          )
                        ],
                      ),
                    ),
                    CircleAvatar(
                      radius: 2.5,
                      backgroundColor:
                          EcommerceAppColor.lightGray.withValues(alpha: 0.3),
                    ),
                    Text(
                      '${product.totalSold} Sold',
                      style: AppTextStyle(context)
                          .bodyTextSmall
                          .copyWith(fontWeight: FontWeight.w500),
                    )
                  ],
                ),
                Gap(10.h),
                Consumer(builder: (context, ref, _) {
                  return Row(
                    mainAxisAlignment: MainAxisAlignment.spaceBetween,
                    children: [
                      Column(
                        children: [
                          if (product.discountPrice > 0) ...[
                            Text(
                              GlobalFunction.price(
                                price: product.discountPrice.toString(),
                                ref: ref,
                              ),
                              style: AppTextStyle(context).bodyText.copyWith(
                                    fontWeight: FontWeight.w700,
                                  ),
                            ),
                          ] else ...[
                            Text(
                              GlobalFunction.price(
                                price: product.price.toString(),
                                ref: ref,
                              ),
                              style: AppTextStyle(context)
                                  .bodyText
                                  .copyWith(fontWeight: FontWeight.w700),
                            ),
                          ],
                          Visibility(
                            visible: product.discountPrice > 0,
                            child: Text(
                              GlobalFunction.price(
                                price: product.price.toString(),
                                ref: ref,
                              ),
                              style: AppTextStyle(context).bodyText.copyWith(
                                    color: EcommerceAppColor.lightGray,
                                    decoration: TextDecoration.lineThrough,
                                    decorationColor:
                                        EcommerceAppColor.lightGray,
                                  ),
                            ),
                          )
                        ],
                      ),
                      IncrementButton(
                        onTap: () {
                          showModalBottomSheet(
                            isDismissible: false,
                            isScrollControlled: true,
                            barrierColor: colors(context)
                                .accentColor!
                                .withValues(alpha: 0.8),
                            shape: RoundedRectangleBorder(
                              borderRadius: BorderRadius.circular(16.r),
                            ),
                            context: context,
                            builder: (_) => AddToCartBottomSheet(
                              product: product,
                            ),
                          );
                        },
                      ),
                    ],
                  );
                })
              ],
            ),
            // if (product.quantity == 0)
            //   Container(
            //     decoration: BoxDecoration(
            //       color:
            //        colors(context).accentColor!.withOpacity(0.3),
            //     ),
            //   ),
          ],
        ),
      ),
    );
  }
}
